<?php
/**
 * Plugin Name: USDTIFY — Accept Payments (Shortcode + WooCommerce)
 * Description: Accept USDTIFY Debit Card payments. Creates intents on your USDTIFY server, renders QR, polls status, marks WooCommerce orders paid via webhook or client-side fallback.
 * Version:     1.4.0
 * Author:      you
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('USDTIFY_Accept_Payments_Full')):

final class USDTIFY_Accept_Payments_Full {
  private static $inst;
  const OPT='usdtify_accept_settings';

  public static function boot(){ return self::$inst ?: (self::$inst=new self()); }

  private function __construct(){
    register_activation_hook(__FILE__, [$this,'on_activate']);
    add_action('admin_menu',        [$this,'menu']);
    add_action('rest_api_init',     [$this,'routes']);
    add_action('wp_enqueue_scripts',[$this,'assets']);
    add_shortcode('usdtify_pay',    [$this,'sc_pay']);
  }

  /* ---------- Activation ---------- */
  private function ensure_page($slug, $title, $content=''){
    $page = get_page_by_path($slug);
    if($page && !is_wp_error($page) && $page->ID){ return get_permalink($page->ID); }
    $id = wp_insert_post([
      'post_title'   => $title,
      'post_name'    => $slug,
      'post_content' => $content,
      'post_status'  => 'publish',
      'post_type'    => 'page',
      'comment_status' => 'closed',
      'ping_status'    => 'closed',
    ], true);
    if(is_wp_error($id) || !$id) return home_url('/');
    return get_permalink($id);
  }

  public function on_activate(){
    $thank_url = $this->ensure_page('usdtify-thank-you', 'USDTIFY — Thank You', '<p>Your payment was successful.</p>');
    $fail_url  = $this->ensure_page('usdtify-failed',    'USDTIFY — Payment Failed', '<p>Your payment did not complete. Please try again.</p>');

    $defaults = [
      'api_key'     => '',
      'base'        => 'https://usdtify.com/wp-json',
      'webhook'     => rest_url('usdtify-accept/v1/webhook'),
      'success_url' => $thank_url,
      'fail_url'    => $fail_url,
    ];
    $cur = get_option(self::OPT, []);
    if(!is_array($cur)) $cur = [];

    $merged = array_merge($defaults, $cur);
    $merged['webhook'] = rest_url('usdtify-accept/v1/webhook');
    if(empty($cur['success_url']) || strpos((string)$cur['success_url'],'usdtify-thank-you')===false) $merged['success_url'] = $thank_url;
    if(empty($cur['fail_url'])    || strpos((string)$cur['fail_url']   ,'usdtify-failed')===false)    $merged['fail_url']    = $fail_url;
    if(empty($cur['base'])) $merged['base'] = 'https://usdtify.com/wp-json';

    update_option(self::OPT, $merged);
  }

  /* ---------- Helpers ---------- */
  private function opt($k,$def=''){ $o=get_option(self::OPT,[]); return array_key_exists($k,$o)?$o[$k]:$def; }

  /* ---------- Admin ---------- */
  public function menu(){
    add_options_page('USDTIFY Accept','USDTIFY Accept','manage_options','usdtify-accept',[$this,'settings']);
  }

  public function settings(){
    if(!current_user_can('manage_options')) return;
    if(isset($_POST['save'])){
      check_admin_referer('usdtify_accept');
      update_option(self::OPT,[
        'api_key'     => sanitize_text_field((string)($_POST['api_key']??'')),
        'base'        => rtrim(esc_url_raw((string)($_POST['base']??'')),'/'),
        'webhook'     => esc_url_raw((string)($_POST['webhook']??'')),
        'success_url' => esc_url_raw((string)($_POST['success_url']??'')),
        'fail_url'    => esc_url_raw((string)($_POST['fail_url']??'')),
      ]);
      echo '<div class="updated"><p>Saved.</p></div>';
    }
    $api  = esc_attr($this->opt('api_key'));
    $base = esc_attr($this->opt('base','https://usdtify.com/wp-json'));
    $wh   = esc_attr($this->opt('webhook', rest_url('usdtify-accept/v1/webhook')));
    $ok   = esc_attr($this->opt('success_url'));
    $ko   = esc_attr($this->opt('fail_url'));

    echo '<div class="wrap"><h1>USDTIFY — Accept Payments</h1><form method="post">';
    wp_nonce_field('usdtify_accept');
    echo '<table class="form-table">
      <tr><th>USDTIFY Base URL</th><td><input type="url" name="base" class="regular-text" value="'.$base.'" placeholder="https://usdtify.com/wp-json"></td></tr>
      <tr><th>Merchant API Key</th><td><input type="text" name="api_key" class="regular-text" value="'.$api.'" placeholder="64-char key"></td></tr>
      <tr><th>Webhook Endpoint</th><td><input type="url" name="webhook" class="regular-text" value="'.$wh.'" readonly></td></tr>
      <tr><th>Success URL</th><td><input type="url" name="success_url" class="regular-text" value="'.$ok.'"></td></tr>
      <tr><th>Fail URL</th><td><input type="url" name="fail_url" class="regular-text" value="'.$ko.'"></td></tr>
    </table>
    <p><button class="button button-primary" name="save" value="1">Save</button></p></form>
    <h2>Shortcode</h2><code>[usdtify_pay]</code></div>';
  }

  /* ---------- Assets ---------- */
  public function assets(){
    // Provide a QR library. Place your minified library at /assets/qrcode.min.js (supports new QRCode.toCanvas OR old QRCode constructor)
    wp_register_script('usdtify-qrcode', plugins_url('assets/qrcode.min.js', __FILE__), [], '2.0.0', false);
  }

  /* ---------- REST (Webhook + Fallback Mark) ---------- */
  public function routes(){
    register_rest_route('usdtify-accept/v1','/webhook',[
      'methods'=>['POST'],'permission_callback'=>'__return_true','callback'=>[$this,'webhook_cb']
    ]);
    register_rest_route('usdtify-accept/v1','/mark',[
      'methods'=>['POST'],'permission_callback'=>'__return_true','callback'=>[$this,'mark_paid_cb']
    ]);
  }

  public function webhook_cb(\WP_REST_Request $r){
    $api=$this->opt('api_key'); if(!$api) return new \WP_Error('cfg','Missing API key',400);
    $sig=(string)$r->get_header('USDTIFY-Signature'); $raw=$r->get_body(); $j=json_decode($raw,true);
    if(!$j||!$sig) return new \WP_Error('bad','Invalid payload',400);

    $msg=(string)($j['event']??'').'|'.(string)($j['intent']??'').'|'.(string)($j['amount']??'').'|'.(string)($j['ts']??'');
    $exp=hash_hmac('sha256',$msg,$api);
    if(!hash_equals($exp,$sig)) return new \WP_Error('sig','Bad signature',401);

    do_action('usdtify_accept_webhook',$j);

    if(!empty($j['event'])&&$j['event']==='pos.paid'&&!empty($j['intent'])&&!empty($j['memo'])&&class_exists('WC_Order')){
      if(preg_match('/WC#(\d+)/',(string)$j['memo'],$m)){
        $oid=(int)$m[1]; if($oid){
          $order=wc_get_order($oid);
          if($order && $order->get_status()!=='completed'){
            $order->payment_complete((string)$j['intent']);
            $net = isset($j['amount_net']) ? $j['amount_net'] : 'n/a';
            $order->add_order_note('USDTIFY paid — Intent '.(string)$j['intent'].'; Net: '.$net);
          }
        }
      }
    }
    return ['ok'=>true];
  }

  public function mark_paid_cb(\WP_REST_Request $r){
    $oid    = (int)($r->get_param('oid') ?? 0);
    $intent = sanitize_text_field((string)($r->get_param('intent') ?? ''));
    if(!$oid || !$intent || !class_exists('WC_Order')) return new \WP_Error('bad','Invalid',400);

    $base = rtrim($this->opt('base','https://usdtify.com/wp-json'),'/');
    $status_url = esc_url_raw($base.'/usdtify/v1/pos/intent/'.$intent.'/status');

    $res = wp_remote_get($status_url, ['timeout'=>15,'headers'=>['Accept'=>'application/json']]);
    if(is_wp_error($res)) return new \WP_Error('net','Status fetch failed',502);

    $code = (int)wp_remote_retrieve_response_code($res);
    $body = json_decode((string)wp_remote_retrieve_body($res), true);
    if($code!==200 || empty($body['status']) || $body['status']!=='paid')
      return new \WP_Error('np','Not paid',409);

    $order = wc_get_order($oid);
    if(!$order) return new \WP_Error('no','Order not found',404);

    if($order->get_status()!=='completed'){
      $order->payment_complete($intent);
      $order->add_order_note('USDTIFY paid (fallback) — Intent '.$intent);
    }
    return ['ok'=>true];
  }

public function sc_pay($atts){
    // shortcode attrs (no defaults -> avoid accidental fallbacks)
    $a = shortcode_atts(['amount'=>'','memo'=>'','order_id'=>''], $atts, 'usdtify_pay');

    // URL overrides (?amount=12.5&memo=Order%20123&order_id=123)
    if (isset($_GET['amount'])   && $_GET['amount']   !== '') $a['amount']   = (string) wp_unslash($_GET['amount']);
    if (isset($_GET['memo'])     && $_GET['memo']     !== '') $a['memo']     = (string) wp_unslash($_GET['memo']);
    if (isset($_GET['order_id']) && $_GET['order_id'] !== '') $a['order_id'] = (string) wp_unslash($_GET['order_id']);

    // Require a valid amount; if missing/invalid, render nothing (no defaults/QR)
    if ($a['amount'] === '' || !is_numeric($a['amount']) || (float)$a['amount'] <= 0) {
      return '';
    }

    $base = rtrim($this->opt('base','https://usdtify.com/wp-json'),'/');
    $api  = $this->opt('api_key');
    if(!$base || !$api) return ''; // silently skip if not configured

    $amount = (float) $a['amount'];
    $memo   = sanitize_text_field((string)$a['memo']);
    if(!empty($a['order_id'])) $memo = trim($memo.' WC#'.intval($a['order_id']));

    $endpoint = $base.'/usdtify/v1/ext/intent';
    $ts = time(); $nonce = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : uniqid('usdtify_', true);

    $res = wp_remote_post($endpoint,[
      'timeout'=>20,
      'headers'=>[
        'Content-Type'=>'application/json',
        'X-API-Key'=>$api,
        'X-API-Ts'=>$ts,
        'X-API-Nonce'=>$nonce,
      ],
      'body'=>wp_json_encode(['amount'=>$amount,'currency'=>'USD','memo'=>$memo,'terminal_label'=>'WP'])
    ]);
    if(is_wp_error($res)) return '';
    $code=(int)wp_remote_retrieve_response_code($res);
    $body=json_decode((string)wp_remote_retrieve_body($res),true);
    if($code!==200||empty($body['ok'])||empty($body['qr_payload'])||empty($body['intent'])) return '';

    $intent   = sanitize_text_field((string)$body['intent']);
    $payloadJS= wp_json_encode($body['qr_payload']);
    $pubStatus= esc_url($base.'/usdtify/v1/pos/intent/'.$intent.'/status');
    $okURL    = esc_url($this->opt('success_url'));
    $koURL    = esc_url($this->opt('fail_url'));
    $markURL  = esc_url(rest_url('usdtify-accept/v1/mark'));
    $oid      = (int)$a['order_id'];

    wp_enqueue_script('usdtify-qrcode');

    $inline = '(function(){' .
      'var root=document.getElementById("usdtifyPay"); if(!root) return;' .
      'var statusURL=root.getAttribute("data-status"); var okURL=root.getAttribute("data-ok"); var koURL=root.getAttribute("data-ko");' .
      'var markURL=root.getAttribute("data-mark"); var oid=parseInt(root.getAttribute("data-oid")||"0",10);' .
      'var qrBox=document.getElementById("qrBox"); var uMsg=document.getElementById("uMsg");' .
      'var payload='.($payloadJS?$payloadJS:'null').';' .
      'var intent="'.esc_js($intent).'";' .
      'function renderQR(){ if(!qrBox) return; var payloadStr=typeof payload==="string"?payload:JSON.stringify(payload||{}); var size=qrBox.getBoundingClientRect().width||320; var dpr=Math.min(2,Math.max(1,window.devicePixelRatio||1)); var px=Math.floor(size*dpr); qrBox.innerHTML=""; if(window.QRCode && typeof window.QRCode.toCanvas==="function"){ var canvas=document.createElement("canvas"); canvas.style.width=size+"px"; canvas.style.height=size+"px"; qrBox.appendChild(canvas); try{ window.QRCode.toCanvas(canvas, payloadStr, {errorCorrectionLevel:"H", width:px, margin:4}, function(err){ if(err&&uMsg) uMsg.textContent="QR render error."; }); }catch(e){ if(uMsg) uMsg.textContent="QR render error."; } return;} if(typeof window.QRCode==="function"){ var holder=document.createElement("div"); holder.style.width=size+"px"; holder.style.height=size+"px"; qrBox.appendChild(holder); try{ new window.QRCode(holder,{text:payloadStr,width:px,height:px,correctLevel:(window.QRCode.CorrectLevel?window.QRCode.CorrectLevel.H:1)}); }catch(e){ if(uMsg) uMsg.textContent="QR render error."; } return;} if(uMsg) uMsg.textContent="QR lib missing.";} ' .
      'renderQR(); window.addEventListener("resize", renderQR);' .
      'var tries=0,max=180; var poll=setInterval(function(){ tries++; if(tries>max){ clearInterval(poll); if(uMsg) uMsg.textContent="Timed out."; window.location.href=koURL; return; } fetch(statusURL,{credentials:"omit"}).then(function(r){return r.json();}).then(function(j){ if(j&&j.status==="paid"){ clearInterval(poll); if(uMsg) uMsg.textContent="Paid. Finalizing..."; if(markURL && oid>0){ fetch(markURL,{method:"POST",headers:{"Content-Type":"application/json"},body:JSON.stringify({oid:oid,intent:intent})}).finally(function(){ setTimeout(function(){ window.location.href=okURL; },600); }); } else { setTimeout(function(){ window.location.href=okURL; },600); } } }).catch(function(){}); },2000);' .
    '})();';

    wp_add_inline_script('usdtify-qrcode', $inline, 'after');

    ob_start(); ?>
<div id="usdtifyPay"
     data-status="<?php echo $pubStatus; ?>"
     data-ok="<?php echo $okURL; ?>"
     data-ko="<?php echo $koURL; ?>"
     data-mark="<?php echo $markURL; ?>"
     data-oid="<?php echo (int)$oid; ?>"
     style="max-width:560px;margin:18px auto;padding:16px;border:1px solid #e2e8f0;border-radius:12px;background:#fff">
  <h3 style="margin:0 0 10px">Pay with USDTIFY</h3>
  <p style="margin:0 0 10px;opacity:.8">Amount: <b><?php echo number_format_i18n($amount,2); ?> USD</b></p>
  <div style="display:grid;place-items:center">
    <div id="qrBox" style="width:320px;height:320px;background:#fff;border:1px solid #d9d9d9;border-radius:10px;display:grid;place-items:center;overflow:hidden"></div>
  </div>
  <p id="uMsg" style="margin:10px 0 0;opacity:.8">Scan with your USDTIFY app to pay.</p>
</div>
<?php
    return ob_get_clean();
  }

}
USDTIFY_Accept_Payments_Full::boot();
endif;

/* ---------- WooCommerce Gateway (standalone) ---------- */
add_action('plugins_loaded','usdtify_register_wc_gateway',20);
function usdtify_register_wc_gateway(){
  if(!class_exists('WooCommerce')||!class_exists('WC_Payment_Gateway')) return;
  if(class_exists('WC_Gateway_USDTIFY')) return;

  class WC_Gateway_USDTIFY extends WC_Payment_Gateway {
    public function __construct(){
      $this->id='usdtify'; $this->method_title='USDTIFY'; $this->method_description='Pay with USDTIFY Debit Card (QR).';
      $this->has_fields=false; $this->supports=['products'];
      $this->init_form_fields(); $this->init_settings();
      $this->title=$this->get_option('title','USDTIFY'); $this->description=$this->get_option('description','Pay securely via USDTIFY.');
      add_action('woocommerce_update_options_payment_gateways_'.$this->id,[$this,'process_admin_options']);
      add_action('woocommerce_receipt_'.$this->id,[$this,'receipt_page']);
      add_action('usdtify_accept_webhook',[$this,'webhook_handler']);
    }
    public function init_form_fields(){
      $this->form_fields=[
        'enabled'=>['title'=>'Enable/Disable','type'=>'checkbox','label'=>'Enable USDTIFY payments','default'=>'yes'],
        'title'=>['title'=>'Title','type'=>'text','default'=>'USDTIFY'],
        'description'=>['title'=>'Description','type'=>'textarea','default'=>'Pay securely via USDTIFY.'],
      ];
    }
    public function process_payment($order_id){
      $order=wc_get_order($order_id); if(!$order) return ['result'=>'fail'];
      return ['result'=>'success','redirect'=>$order->get_checkout_payment_url(true)];
    }
    public function receipt_page($order_id){
      $order=wc_get_order($order_id); if(!$order){ echo '<p>Error.</p>'; return; }
      $amount=(float)$order->get_total();
      $memo='WC Order #'.$order->get_order_number().' ('.$order->get_id().')';
      echo do_shortcode('[usdtify_pay amount="'.esc_attr($amount).'" memo="'.esc_attr($memo).'" order_id="'.esc_attr($order->get_id()).'"]');
    }
    public function webhook_handler($j){
      if(empty($j['event'])||$j['event']!=='pos.paid'||empty($j['memo'])) return;
      if(preg_match('/WC#(\d+)/',(string)$j['memo'],$m)){
        $oid=(int)$m[1]; if(!$oid) return; $order=wc_get_order($oid); if(!$order) return;
        if($order->get_status()!=='completed'){
          $order->payment_complete((string)($j['intent']??'')); 
          $order->add_order_note('USDTIFY paid — Intent '.((string)($j['intent']??'')).'; Net: '.((string)($j['amount_net']??'n/a')));
          if(function_exists('WC') && isset(WC()->cart) && WC()->cart){ WC()->cart->empty_cart(); }
        }
      }
    }
  }
  add_filter('woocommerce_payment_gateways', function($g){ $g[]='WC_Gateway_USDTIFY'; return $g; });
}

/* Ensure: 1) /assets/qrcode.min.js exists; 2) theme calls wp_head() and wp_footer() */
